/* Copyright (C) 2011-2018 RealVNC Ltd.  All Rights Reserved.
*/

#ifndef __VNCDISCOVERYCALLBACKS_H__
#define __VNCDISCOVERYCALLBACKS_H__

/**
 * \file vncdiscoverysdkcallbacks.h
 *
 * This file defines types for pointers to callback functions that may be
 * provided to the VNC Discovery SDK. You should normally include 
 * vncdiscoverysdk.h instead of including this file directly.
 *
 * All VNC Discoverer callbacks, except logging callbacks, are invoked by the
 * Discovery SDK thread.
 *
 * The client application needs to create and populate a
 * VNCDiscoverySDKCallbacks structure, but it can choose not to receive some of
 * the callbacks. This can be done by keeping the function pointer related to a
 * certain callback to NULL.
 *
 * \see VNCDiscoverySDKCallbacks, VNCDiscoverySDK
 * \see vncdiscoverysdk.h
 */

#include <vncdiscoverysdktypes.h>

#ifdef __cplusplus
extern "C"
{
#endif

/**
 * \brief Notifies of the start of the SDK thread.
 *
 * Notifies the client that the Discovery SDK started a new thread. 
 *
 * Callbacks made by the SDK will happen only on threads that were reported
 * through this callback.
 *
 * \param pSDKContext The context associated with the SDK.
 */
typedef void VNCCALL
VNCDiscoverySDKThreadStartedCallback(void *pSDKContext);

/**
 * \brief Notifies that the SDK thread is about to end.
 *
 * Notifies the client that a Discovery SDK thread is about to end.
 *
 * No further callbacks will happen from this thread.
 *
 * \param pSDKContext The context associated with the SDK.
 * \param error The final outcome error of the thread.
 */
typedef void VNCCALL
VNCDiscoverySDKThreadEndedCallback(void *pSDKContext, 
                                  VNCDiscoverySDKError error);

/**
 * \brief Notifies the client of the outcome of Start Discovering.
 *
 * Informs the client whether the Start operation on the Discoverer
 * instance was successful or not.
 *
 * \param pSDKContext The SDK context that was set by the client application for
 *                 callbacks.
 * \param pDiscoverer The Discoverer that started.
 * \param error The outcome of the start request. VNCDiscoverySDKErrorNone if
 *              the request was successful.
 *              VNCDiscoverySDKErrorInvalidParameter if the Discoverer passed
 *              in was NULL, VNCDiscoverySDKErrorOutOfMemory if there is not
 *              enough memory to complete the request, or another
 *              Discoverer-specific error code.
 */
typedef void VNCCALL
VNCDiscoverySDKDiscovererStartedCallback(void *pSDKContext,
                                  VNCDiscoverySDKDiscoverer *pDiscoverer, 
                                  VNCDiscoverySDKError error);

/**
 * \brief Notifies the client of the outcome of Stop Discovering.
 *
 * Informs the client that the Discoverer instance was stopped.
 *
 * \param pSDKContext The SDK context that was set by the client application for
 *                 callbacks.
 * \param pDiscoverer The Discoverer that stopped.
 * \param error The outcome of the stop request. VNCDiscoverySDKErrorNone if
 *              the request was successful, or another
 *              Discoverer-specific error code.
 */
typedef void VNCCALL
VNCDiscoverySDKDiscovererStoppedCallback(void *pSDKContext,
                                  VNCDiscoverySDKDiscoverer *pDiscoverer,
                                  VNCDiscoverySDKError error);

/**
 * \brief Notifies the client that an entity has appeared.
 *
 * Notifies the client that an Entity has appeared for a specific Discoverer.
 * The entity remains under the ownership of the Discoverer and will be freed 
 * when either the entity disappears, or when the Discoverer instance is 
 * destroyed.
 *
 * \note The existence of an entity is not definite. It is not guaranteed that a VNC
 * Server is present and listening for incoming connections. False positives
 * might be reported by some Discoverers.
 *
 * Requests can be made for this entity as long as the Discoverer is running.
 *
 * \param pSDKContext The SDK context that was set by the client application for
 *                 callbacks.
 * \param pDiscoverer The Discoverer that discovered the entity.
 * \param pEntity The entity that appeared.
 */
typedef void VNCCALL
VNCDiscoverySDKEntityAppearedCallback(void *pSDKContext,
                                    VNCDiscoverySDKDiscoverer *pDiscoverer,
                                    VNCDiscoverySDKEntity *pEntity);

/**
 * \brief Notifies the client that an entity has changed.
 *
 * Notifies the client that an Entity has changed for a specific Discoverer.
 *
 * Not all Discoverers support this call, so check the Discoverer-specific
 * notification to find out if the client of the SDK would be notified of
 * changes to entities.
 *
 * If a Discoverer notifies of changes to the entities, you need to read the
 * Discoverer documentation to understand the meaning of the callback. Some
 * Discoverer might mean that you need to re-query everything about the entity
 * when it notifies of a change, while another Discoverer might mean only that
 * some aspect of the entity has changed.
 *
 * \param pSDKContext The SDK context that was set by the client application for
 *                 callbacks.
 * \param pDiscoverer The Discoverer that discovered the entity.
 * \param pEntity The entity that changed.
 * \param pChangeDescription A string that describes what has changed. Can be
 * NULL. Ownership of the string stays with the SDK.
 */
typedef void VNCCALL
VNCDiscoverySDKEntityChangedCallback(void *pSDKContext,
                                    VNCDiscoverySDKDiscoverer *pDiscoverer,
                                    VNCDiscoverySDKEntity *pEntity,
                                    const char *pChangeDescription);

/**
 * \brief Notifies the client that an entity has disappeared.
 *
 * Notifies the client that an Entity has disappeared for a specific Discoverer.
 *
 * The memory allocated for this entity will be freed on returning from this
 * callback.
 *
 * Requests can no longer be made by the client of the SDK.
 *
 * \param pSDKContext The SDK context that was set by the client application for
 *                 callbacks.
 * \param pDiscoverer The Discoverer that discovered the entity.
 * \param pEntity The entity that disappeared.
 */
typedef void VNCCALL
VNCDiscoverySDKEntityDisappearedCallback(void *pSDKContext,
                                      VNCDiscoverySDKDiscoverer *pDiscoverer,
                                      VNCDiscoverySDKEntity *pEntity);

/**
 * \brief Returns the results of an asynchronous fetch request.
 *
 * Notifies the client that an Entity Value fetch call has completed and
 * returns details of the relevant Discoverer, Entity and Key-Value pair.
 * 
 * If an error occurred it will be reported in the error parameter and the 
 * pValue parameter may be NULL.
 *
 * \note This callback can happen before VNCDiscoverySDKFetchEntityValue()
 * returned, so the requestId might be a new one.
 *
 * \param pSDKContext The SDK context that was set by the client application for
 *                 callbacks.
 * \param requestId The ID of the request. This was given to the client
 *                  application when the application made the request
 *                  initially.
 * \param pDiscoverer The Discoverer that discovered the entity.
 * \param pEntity The entity whose value has been retrieved.
 * \param pKey The key that was requested. This string is owned by the client
 *             SDK once this call is made, and must be freed by calling
 *             VNCDiscoverySDKFreeString() when it is no longer needed. The
 *             string can be NULL if an Out of Memory error occurred.
 * \param error The outcome of the request: VNCDiscoverySDKErrorNone if the
 *              request was completed successfully,
 *              VNCDiscoverySDKErrorNotSupported if the key is not supported by
 *              the Entity, VNCDiscoverySDKErrorOutOfMemory if there is not
 *              enough memory to complete the request,
 *              VNCDiscoverySDKErrorInvalidParameter if the Entity or Key was 
 *              NULL, VNCDiscoverySDKErrorUnknownEntity if the Entity doesn't exist, 
 *              VNCDiscoverySDKErrorCancelled if the Discoverer was stopped
 *              while processing the request, or another error specific to the Discoverer.
 * \param pValue The value requested, corresponding to the key. This string is
 *               owned by the client of the SDK, and must be freed by calling
 *               VNCDiscoverySDKFreeString() when it is no longer needed.
 *               This can be NULL if some error occurred.
 *
 * \see VNCDiscoverySDKFetchEntityValue
 */
typedef void VNCCALL
VNCDiscoverySDKFetchEntityValueCallback(void *pSDKContext,
                                      VNCDiscoverySDKRequestId requestId, 
                                      VNCDiscoverySDKDiscoverer *pDiscoverer,
                                      VNCDiscoverySDKEntity *pEntity,
                                      char *pKey,
                                      VNCDiscoverySDKError error,
                                      char *pValue);

/**
 * \brief Notifies the outcome of an asynchronous post request.
 *
 * Notifies the client that an Entity post call has completed and
 * returns details of the relevant Discoverer, Entity and Key.
 * 
 * If an error occurred it will be reported in the error parameter.
 *
 * \param pSDKContext The SDK context that was set by the client application for
 *                 callbacks.
 * \param requestId The ID of the request. This was given to the client
 *                  application when the application made the request
 *                  initially.
 * \param pDiscoverer The Discoverer that discovered the entity.
 * \param pEntity The entity whose value has been set.
 * \param pKey The key that was requested. This string is owned by the client
 *             SDK once this call is made, and must be freed by calling
 *             VNCDiscoverySDKFreeString() when it is no longer needed.
 * \param pValue The value requested, corresponding to the key. This string is
 *               owned by the client of the SDK, and must be freed by calling
 *               VNCDiscoverySDKFreeString() when it is no longer needed.
 *               This can be NULL if some error occurred.
 * \param error The outcome of the request: VNCDiscoverySDKErrorNone if the
 *              request was completed successfully,
 *              VNCDiscoverySDKErrorNotSupported if the key is not supported by
 *              the Entity, or is read-only, VNCDiscoverySDKErrorOutOfMemory if 
 *              there is not enough memory to complete the request,
 *              VNCDiscoverySDKErrorInvalidParameter if the Discoverer, Entity
 *              or Key was NULL, or the value was set to NULL when the Entity
 *              doesn't support un-setting of the key, 
 *              VNCDiscoverySDKErrorUnknownEntity if the Entity doesn't exist, or 
 *              VNCDiscoverySDKErrorCancelled if the Discoverer was stopped
 *              while processing the request, or another error specific to the Discoverer.
 */
typedef void VNCCALL
VNCDiscoverySDKPostEntityValueCallback(void *pSDKContext,
                                      VNCDiscoverySDKRequestId requestId, 
                                      VNCDiscoverySDKDiscoverer *pDiscoverer,
                                      VNCDiscoverySDKEntity *pEntity,
                                      char *pKey,
                                      char *pValue,
                                      VNCDiscoverySDKError error);

/**
 * \brief Notifies of the outcome of a publish entity call.
 *
 * This is called by address-book like Discoverers to notify the outcome of a
 * publish entity request.
 *
 * \param pSDKContext The SDK context that was set by the client application for
 *                 callbacks.
 * \param pDiscoverer The Discoverer that owns the entity.
 * \param pEntity The entity that was requested to be published.
 * \param error The outcome of the publish: VNCDiscoverySDKErrorNone if the
 *              request was successful, VNCDiscoverySDKErrorUnknownEntity if the
 *              entity can not be found, VNCDiscoverySDKErrorOutOfMemory if
 *              there wasn't enough memory to complete the request,
 *              VNCDiscoverySDKErrorInvalidParameter if the Discoverer, or
 *              Entity was NULL, VNCDiscoverySDKErrorCancelled if the
 *              Discoverer was stopped while processing the request, 
 *              or another Discoverer-specific error code.
 */
typedef void VNCCALL
VNCDiscoverySDKEntityPublishedCallback(void *pSDKContext,
                                VNCDiscoverySDKDiscoverer *pDiscoverer,
                                VNCDiscoverySDKEntity *pEntity,
                                VNCDiscoverySDKError error);

/**
 * \brief Notifies of the outcome of an unpublish entity call.
 *
 * This is called by address-book like Discoverers to notify the outcome of an
 * unpublish entity request.
 *
 * \param pSDKContext The SDK context that was set by the client application for
 *                 callbacks.
 * \param pDiscoverer The Discoverer that owns the entity.
 * \param pEntity The entity that was requested to be unpublished.
 * \param error The outcome of the publish: VNCDiscoverySDKErrorNone if the
 *              request was successful, VNCDiscoverySDKErrorUnknownEntity if the
 *              entity can not be found, VNCDiscoverySDKErrorOutOfMemory if
 *              there wasn't enough memory to complete the request,
 *              VNCDiscoverySDKErrorInvalidParameter if the Discoverer, or
 *              Entity was NULL, VNCDiscoverySDKErrorCancelled if the
 *              Discoverer was stopped while processing the request, or another Discoverer-specific error code.
 */
typedef void VNCCALL
VNCDiscoverySDKEntityUnpublishedCallback(void *pSDKContext,
                                VNCDiscoverySDKDiscoverer *pDiscoverer,
                                VNCDiscoverySDKEntity *pEntity,
                                VNCDiscoverySDKError error);

/**
 * \brief Sends a log message to the client application.
 *
 * This is called by the SDK with logging information. There is no guarantee on
 * which thread this callback occurs, but only one thread will issue the
 * callback at any one time. So while the logging callback is handled in a
 * thread, no other thread will issue another logging callback.
 *
 * If this method is not implemented by the client application, no logging will
 * be made.
 *
 * The application should only take the necessary actions to log the message.
 * No other action, like calling back into the Discovery SDK, should be taken.
 *
 * \param pSDKContext The SDK context that was set by the client application
 *              for callbacks.
 * \param severity The severity of the log. For an explanation of the severity
 *              levels see VNCDiscoverySDKSetLoggingSeverity.
 * \param pDiscoverer The name of the Discoverer that outputs the logging. If
 *              it's an empty string then the logging originates from the
 *              Discovery SDK itself.
 * \param pText The text that is logged.
 *
 * \see VNCDiscoverySDKSetLoggingSeverity
 */
typedef void VNCCALL
VNCDiscoverySDKLogCallback(void *pSDKContext, 
                                vnc_int32_t severity,
                                const char *pDiscoverer, 
                                const char *pText);

/**
 * \brief Notifies the client when the status of a Discoverer changes.
 *
 * Tells the client that the status of the Discoverer has changed. The status
 * can be one of the following:
 *  - VNCDiscoverySDKDiscovererStatusWaiting if the Discoverer does not have
 *  any devices that need scanning for new entities. No new devices have been 
 *  detected. The Discoverer is waiting for new devices to become available.
 *  - VNCDiscoverySDKDiscovererStatusScanning if the Discoverer is currently
 *  scanning one or more devices. These devices have become available recently
 *  and the Discoverer is scanning them for new entities.
 *
 * \param pSDKContext The SDK context that was set by the client application for
 *                 callbacks.
 * \param pDiscoverer The Discoverer that changed status.
 * \param status The new status.
 */
typedef void VNCCALL
VNCDiscoverySDKDiscovererStatusUpdateCallback(void *pSDKContext,
                                    VNCDiscoverySDKDiscoverer *pDiscoverer,
                                    VNCDiscoverySDKDiscovererStatus status);

/**
 * \brief Asks the client application to choose which Discoverer will handle
 * a device.
 *
 * This is called when the SDK can not establish which Discoverer should
 * receive a device. The client application may ask the user, or decide by
 * itself automatically, and report the decision through
 * VNCDiscoverySDKOfferDeviceToDiscoverer.
 *
 * If the device becomes unavailable, VNCDiscoverySDKCancelDeviceChoice is
 * called.
 *
 * \note When the parameter VNCDiscoverySDKParameterAlwaysAskAboutDevice is
 * set to false, if a Discoverer has ever been offered a device, then it will
 * not be offered the device again even if it has released the device. But when
 * the parameter VNCDiscoverySDKParameterAlwaysAskAboutDevicethis is set to true,
 * if a Discoverer has ever been selected to be offered a device, then the
 * Discoverer will be included in this callback again once it has released the
 * device.
 *
 * \param pSDKContext The SDK context that was set by the client application
 * for callbacks.
 * \param pDevice The device that should be checked. Ownership stays with the
 * SDK. The device remains present until either
 * VNCDiscoverySDKOfferDeviceToDiscoverer is called, or the
 * VNCDiscoverySDKCancelDeviceChoice callback is received.
 * \param ppDiscoverersRequestingAccess Information regarding what Discoverers
 * are requesting access. A NULL-terminated array of pointers. Ownership
 * remains with the SDK and the array and structures are valid until a call to
 * either VNCDiscoverySDKOfferDeviceToDiscoverer, or
 * VNCDiscoverySDKCancelDeviceChoice.
 *
 * \see VNCDiscoverySDKOfferDeviceToDiscoverer, VNCDiscoverySDKDevice,
 * VNCDiscovererRequestingAccess, VNCDiscoverySDKCancelDeviceChoice
 */
typedef void VNCCALL
VNCDiscoverySDKChooseDiscovererForDevice(void *pSDKContext,
                    const VNCDiscoverySDKDevice *pDevice,
                    const VNCDiscovererRequestingAccess * const *ppDiscoverersRequestingAccess);

/**
 * \brief Cancels the request to choose which Discoverer will handle a device.
 *
 * This is usually called when the device is no longer present, or the
 * Discoverers are not interested in it anymore.
 *
 * Once this is called, the device is no longer valid and the application is
 * not expected to give a response to VNCDiscoverySDKChooseDiscovererForDevice.
 *
 * \param pSDKContext The SDK context that was set by the client application
 * for callbacks.
 * \param pDevice The device for which the request is cancelled.
 *
 * \see VNCDiscoverySDKOfferDeviceToDiscoverer, VNCDiscoverySDKDevice,
 * VNCDiscoverySDKChooseDiscovererForDevice
 */
typedef void VNCCALL
VNCDiscoverySDKCancelDeviceChoice(void *pSDKContext,
                    const VNCDiscoverySDKDevice *pDevice);

/**
 * \brief Notifies the client, or any subscribed listener that the Discoverer is
 * being destroyed.
 *
 * The Discoverer must not be used after returning from this callback.
 *
 * If a listener is receiving this call it shouldn't attempt to remove itself
 * from being a discoverer listener.
 *
 * \param pSDKContext The SDK context that was set by the client application for
 *                 callbacks.
 * \param pDiscoverer The Discoverer that is destroyed.
 *
 * \see ::VNCDiscoverySDKAddDiscovererListener
 */
typedef void VNCCALL
VNCDiscoverySDKDiscovererDestroyedCallback(void *pSDKContext,
                                  VNCDiscoverySDKDiscoverer *pDiscoverer);

/**
 * \brief The structure that the client of the SDK needs to populate to receive
 * callbacks from the Discovery SDK.
 *
 * The client SDK can leave entries for callbacks that are not needed NULL and
 * the SDK will not make those callbacks.
 *
 * As a minimum a client application would want to have the Entity Appeared
 * Callback to hear of any entity that gets discovered. Otherwise it can choose
 * not to be called back for anything else. 
 */
typedef struct
{
  /* Thread callbacks */
  /**
   * Notifies of the start of the SDK thread.
   */
  VNCDiscoverySDKThreadStartedCallback *threadStarted;
  /**
   * Notifies that the SDK thread is about to end.
   */
  VNCDiscoverySDKThreadEndedCallback *threadEnded;
  /* Discoverer state callbacks */
  /**
   * Notifies the client of the outcome of Start Discovering.
   */
  VNCDiscoverySDKDiscovererStartedCallback *discovererStarted;
  /**
   * Notifies the client of the outcome of Stop Discovering.
   */
  VNCDiscoverySDKDiscovererStoppedCallback *discovererStopped;
  /* Entity presence callbacks */
  /**
   * Notifies the client that an entity has appeared.
   */
  VNCDiscoverySDKEntityAppearedCallback *entityAppeared;
  /**
   * Notifies the client that an entity has changed.
   */
  VNCDiscoverySDKEntityChangedCallback *entityChanged;
  /**
   * Notifies the client that an entity has disappeared.
   */
  VNCDiscoverySDKEntityDisappearedCallback *entityDisappeared;
  /* Entity operations callbacks */
  /**
   * Returns the results of an asynchronous fetch request.
   */
  VNCDiscoverySDKFetchEntityValueCallback *fetchEntityValue;
  /**
   * Notifies the outcome of an asynchronous post request.
   */
  VNCDiscoverySDKPostEntityValueCallback *postEntityValue;
  /* Adressbook operations callbacks */
  /**
   * Notifies of the outcome of a publish entity call.
   */
  VNCDiscoverySDKEntityPublishedCallback *entityPublished;
  /**
   * Notifies of the outcome of an unpublish entity call.
   */
  VNCDiscoverySDKEntityUnpublishedCallback *entityUnpublished;
  /**
   * Sends a log message to the client application.
   */
  VNCDiscoverySDKLogCallback *log;
  /**
   * Notifies the client when the status of a Discoverer changes.
   */
  VNCDiscoverySDKDiscovererStatusUpdateCallback *statusUpdate;
  /**
   * Asks the client application to choose which discoverer will handle
   * a device.
   */
  VNCDiscoverySDKChooseDiscovererForDevice *chooseDiscovererForDevice;
  /**
   * Cancels the request to choose which discoverer will handle a device.
   */
  VNCDiscoverySDKCancelDeviceChoice *cancelDeviceChoice;
  /**
   * The context pointer of the client application. The application can
   * associate a pointer for its context with a Discovery SDK instance. The
   * context is passed to the application with each callback. It is recommented
   * to set the context when creating the SDK.
   *
   * \see VNCDiscoverySDKSetContext
   */
  void *pSDKContext;
  /**
   * Notifies the client, or any subscribed listener that the Discoverer is
   * being destroyed.
   */
  VNCDiscoverySDKDiscovererDestroyedCallback *discovererDestroyed;
} VNCDiscoverySDKCallbacks;

#ifdef __cplusplus
}
#endif

#endif /* !defined(__VNCDISCOVERYCALLBACKS_H__) */

